<?php

/*
 * This file is part of the ActiveCollab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

declare(strict_types=1);

namespace ActiveCollab\Foundation\Wrappers\Cache;

use ActiveCollab\Foundation\App\Mode\ApplicationModeInterface;
use Angie\Inflector;
use AngieApplication;
use Closure;
use DataObject;
use InvalidParamError;
use RuntimeException;
use Stash\Driver\Apc;
use Stash\Driver\FileSystem;
use Stash\Driver\FileSystem\SerializerEncoder;
use Stash\Driver\Memcache;
use Stash\Interfaces\ItemInterface;
use Stash\Pool;

class Cache implements CacheInterface
{
    const FILESYSTEM_BACKEND = 'filesystem';
    const MEMCACHED_BACKEND = 'memcached';
    const APC_BACKEND = 'apc';

    private int $lifetime;

    public function __construct(int $lifetime = CacheInterface::DEFAULT_LIFETIME)
    {
        $this->lifetime = $lifetime;
    }

    /**
     * Return true if $key is cached.
     *
     * @param mixed $key
     */
    public function isCached($key): bool
    {
        $stash = $this->getStash($this->getKey($key));
        $stash->get();

        return !$stash->isMiss();
    }

    /**
     * Return value for a given key.
     *
     * @param  string|array $key
     * @param  mixed        $default
     * @return mixed|null
     */
    public function get(
        $key,
        $default = null,
        bool $force_refresh = false,
        int $lifetime = null
    )
    {
        $stash = $this->getStash($this->getKey($key));

        $data = $stash->get();

        if ($force_refresh || $stash->isMiss()) {
            $data = is_callable($default)
                ? call_user_func($default)
                : $default;

            $stash
                ->set($data)
                ->setTTL($this->getLifetime($lifetime));

            $this->pool->save($stash);
        }

        return $data;
    }

    /**
     * Return by object.
     *
     * @param  object|array  $object
     * @param  string        $sub_namespace
     * @param  Closure|mixed $default
     * @param  int           $lifetime
     * @return mixed
     */
    public function getByObject(
        $object,
        $sub_namespace = null,
        $default = null,
        bool $force_refresh = false,
        int $lifetime = null
    )
    {
        if (!$this->isValidObject($object)) {
            throw new InvalidParamError('object', $object, '$object is not a valid cache context');
        }

        return $this->get($this->getCacheKeyForObject($object, $sub_namespace), $default, $force_refresh, $lifetime);
    }

    /**
     * Return true if $object is instance that we can work with.
     *
     * @param object|array $object
     */
    public function isValidObject($object): bool
    {
        if ($object instanceof DataObject) {
            return $object->isLoaded();
        } elseif (is_array($object) && count($object) == 2) {
            return true;
        } else {
            return is_object($object)
                && method_exists($object, 'getId')
                && method_exists($object, 'getModelName')
                && $object->getId();
        }
    }

    /**
     * Cache given value.
     *
     * @param  mixed $key
     * @param  mixed $value
     * @param  mixed $lifetime
     * @return mixed
     */
    public function set($key, $value, $lifetime = null)
    {
        $stash = $this
            ->getStash($this->getKey($key))
                ->set($value)
                ->setTTL($this->getLifetime($lifetime));

        $this->pool->save($stash);

        return $value;
    }

    /**
     * Set value by given object.
     *
     * @param  object|array $object
     * @param  mixed        $sub_namespace
     * @param  mixed        $value
     * @param  int          $lifetime
     * @return mixed
     */
    public function setByObject($object, $sub_namespace, $value, $lifetime = null)
    {
        if ($this->isValidObject($object)) {
            return $this->set($this->getCacheKeyForObject($object, $sub_namespace), $value, $lifetime);
        } else {
            return false; // Not supported for objects that are not persisted
        }
    }

    /**
     * Remove value and all sub-nodes.
     *
     * @param $key
     */
    public function remove($key)
    {
        $this->getStash($key)->clear();
    }

    /**
     * Remove data by given object.
     *
     * $sub_namespace let you additionally specify which part of object's cache should be removed, instead of entire
     * object cache. Example:
     *
     * AngieApplication::cache()->removeByObject($user, 'permissions_cache');
     *
     * @param       $object
     * @param mixed $sub_namespace
     */
    public function removeByObject($object, $sub_namespace = null)
    {
        $this->remove($this->getCacheKeyForObject($object, $sub_namespace));
    }

    public function removeByModel(string $model_name): void
    {
        $this->remove(
            [
                'models',
                $model_name,
            ]
        );
    }

    public function clear(): void
    {
        if ($this->getPool()->getDriver() instanceof FileSystem) {
            empty_dir(CACHE_PATH, true);
        } else {
            $this->getPool()->clear();
        }
    }

    public function clearModelCache(): void
    {
        $this->remove('models');
    }

    public function getCacheKeyForObject($object, $subnamespace = null): array
    {
        if (!$this->isValidObject($object)) {
            throw new InvalidParamError('object', $object, '$object is expected to be loaded object instance with getId method defined or an array that has model name and object ID');
        }

        if ($object instanceof DataObject) {
            return get_data_object_cache_key(
                $object->getModelName(true),
                $object->getId(),
                $subnamespace
            );
        } elseif (is_array($object) && count($object) == 2) {
            return get_data_object_cache_key($object[0], $object[1], $subnamespace);
        } else {
            return get_data_object_cache_key(
                Inflector::pluralize(Inflector::underscore(get_class($object))),
                $object->getId(),
                $subnamespace
            );
        }
    }

    // ---------------------------------------------------
    //  Internal, Stash Related Functions
    // ---------------------------------------------------

    private ?Pool $pool = null;

    private function getPool(): Pool
    {
        if (empty($this->pool)) {
            $this->pool = new Pool();

            $this->pool->setNamespace(
                sprintf(
                    'cachestore%d',
                    AngieApplication::getAccountId()
                )
            );

            switch ($this->getCacheBackend()) {
                case self::MEMCACHED_BACKEND:
                    $this->pool->setDriver($this->getMemcacheDriver());
                    break;
                case self::APC_BACKEND:
                    $this->pool->setDriver($this->getApcDriver());
                    break;
                default:
                    if (!self::allowFileSystemCache()) {
                        throw new RuntimeException('On Demand system cannot use file system cache. Check configuration');
                    }

                    $this->pool->setDriver($this->getFileSystemDriver(CACHE_PATH));
            }
        }

        return $this->pool;
    }

    private function getCacheBackend(): string
    {
        if (defined('CACHE_BACKEND') && CACHE_BACKEND) {
            switch (CACHE_BACKEND) {
                case 'MemcachedCacheBackend':
                case self::MEMCACHED_BACKEND:
                    return self::MEMCACHED_BACKEND;

                case 'APCCacheBackend':
                case self::APC_BACKEND:
                    return self::APC_BACKEND;
            }
        }

        return self::FILESYSTEM_BACKEND;
    }

    private function allowFileSystemCache(): bool
    {
        return AngieApplication::getContainer()->get(ApplicationModeInterface::class)->isInTestMode()
            || AngieApplication::getContainer()->get(ApplicationModeInterface::class)->isInDevelopment()
            || !AngieApplication::isOnDemand();
    }

    private function getStash($key): ItemInterface
    {
        return $this->getPool()->getItem(
            is_array($key) ? implode('/', $key) : $key
        );
    }

    private function getMemcacheDriver(): Memcache
    {
        defined('CACHE_MEMCACHED_SERVERS') or define('CACHE_MEMCACHED_SERVERS', '');

        $prefix_key = defined('CACHE_MEMCACHED_PREFIX') && CACHE_MEMCACHED_PREFIX
            ? CACHE_MEMCACHED_PREFIX
            : AngieApplication::getAccountId() . '-' . APPLICATION_UNIQUE_KEY;

        $driver = new Memcache(
            [
                'servers' => $this->parseMemcachedServersList((string) CACHE_MEMCACHED_SERVERS),
                'prefix_key' => $prefix_key,
            ]
        );

        return $driver;
    }

    private function parseMemcachedServersList(string $list): array
    {
        $result = [];

        if ($list) {
            foreach (explode(',', $list) as $server) {
                if (strpos($server, '/') !== false) {
                    [$server_url, $weight] = explode('/', $server);
                } else {
                    $server_url = $server;
                    $weight = 1;
                }

                $parts = parse_url($server_url);

                if (empty($parts['host'])) {
                    if (empty($parts['path'])) {
                        continue; // Ignore
                    } else {
                        $host = $parts['path'];
                    }
                } else {
                    $host = $parts['host'];
                }

                $result[] = [$host, array_var($parts, 'port', '11211'), $weight];
            }
        }

        return $result;
    }

    private function getApcDriver(string $namespace = null): Apc
    {
        return new Apc(
            [
                'ttl' => $this->getLifetime(),
                'namespace' => empty($namespace) ? md5(APPLICATION_UNIQUE_KEY) : $namespace,
            ]
        );
    }

    private function getFileSystemDriver(string $path): FileSystem
    {
        return new FileSystem(
            [
                'dirSplit' => 1,
                'path' => $path,
                'filePermissions' => 0777,
                'dirPermissions' => 0777,
                'encoder' => new SerializerEncoder(),
            ]
        );
    }

    public function getBackendType(): ?string
    {
        if ($this->pool) {
            if ($this->pool->getDriver() instanceof Memcache) {
                return self::MEMCACHED_BACKEND;
            } elseif ($this->pool->getDriver() instanceof Apc) {
                return self::APC_BACKEND;
            } elseif ($this->pool->getDriver() instanceof FileSystem) {
                return self::FILESYSTEM_BACKEND;
            }
        }

        return null;
    }

    private function getKey($key): string
    {
        return is_array($key)
            ? implode('/', $key)
            : (string) $key;
    }

    private function getLifetime(int $lifetime = null): int
    {
        return $lifetime ? $lifetime : $this->lifetime;
    }
}
