<?php

/*
 * This file is part of the ActiveCollab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

declare(strict_types=1);

use ActiveCollab\ActiveCollabJobs\Jobs\Instance\Job;
use ActiveCollab\ActiveCollabJobs\Jobs\Push\PushNotificationJob;
use ActiveCollab\Foundation\Notifications\Channel\NotificationChannel;
use ActiveCollab\Module\System\Utils\PushNotification\PushNotificationServiceInterface;
use Angie\Notifications\PushNotificationInterface;

class PushNotificationChannel extends NotificationChannel
{
    const CHANNEL_NAME = 'push';

    private PushNotificationServiceInterface $service;

    private $badge_counts = [];

    public function __construct(PushNotificationServiceInterface $service)
    {
        $this->service = $service;
    }

    public function getShortName(): string
    {
        return self::CHANNEL_NAME;
    }

    public function getVerboseName()
    {
        return 'Push Notification Channel';
    }

    public function isEnabledByDefault()
    {
        return true;
    }

    public function isEnabledFor(User $user)
    {
        return true;
    }

    public function getBadgeCount(IUser $recipient): int
    {
        if (isset($this->badge_counts[$recipient->getId()])) {
            return $this->badge_counts[$recipient->getId()];
        }
        $count = $this->service->getManager()->getUnreadNotificationsCountForUserId($recipient->getId());
        $this->badge_counts[$recipient->getId()] = $count;

        return $count;
    }

    public function send(
        Notification &$notification,
        IUser $recipient,
        bool $skip_sending_queue = false
    )
    {
        if (AngieApplication::isOnDemand() && $notification instanceof PushNotificationInterface) {
            $devices = $this->service->getManager()->findDevicesByUserId($recipient->getId());
            if (count($devices) > 0) {
                $tokens = \array_map(function ($device) {
                    return $device->getToken();
                }, $devices);
                $instance_id = AngieApplication::getAccountId();
                $payload = $this->service->getPayload($notification, $recipient);
                $job_data = [
                    'device_tokens' => $tokens,
                    'title' => $payload['title'],
                    'body' => $payload['body'],
                    'instance_id' => $instance_id,
                    'priority' => Job::HAS_HIGHEST_PRIORITY,
                    'attempts' => 2,
                    'data' => $payload['data'],
                    'badge' => $this->getBadgeCount($recipient),
                ];
                $job = new PushNotificationJob($job_data);
                AngieApplication::log()->info('PUSH_NOTIFICATION_PAYLOAD', $job_data);
                AngieApplication::jobs()->dispatch($job, self::CHANNEL_NAME);
            }
        }
    }
}
