<?php

/*
 * This file is part of the ActiveCollab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

declare(strict_types=1);

use Angie\Http\Request;
use Angie\Http\Response;

AngieApplication::useController('conversations', EnvironmentFramework::INJECT_INTO);

class ConversationMessagesController extends ConversationsController
{
    private ?Message $message = null;

    public function __before(Request $request, $user)
    {
        $before_result = parent::__before($request, $user);

        if ($before_result !== null) {
            return $before_result;
        }

        if ($message_id = $request->getId('message_id')) {
            $this->message = DataObjectPool::get(Message::class, $message_id);

            if (!$this->message) {
                return Response::NOT_FOUND;
            }
        }

        return null;
    }

    public function create(Request $request, User $user)
    {
        return $this->conversation->createMessage(
            $user,
            (string) $request->post('body')
        );
    }

    public function index(Request $request, User $user)
    {
        return $this->conversation->getMessages($user);
    }

    public function view(Request $request, User $user)
    {
        if (!$this->message->canView($user)) {
            return Response::FORBIDDEN;
        }

        return $this->message;
    }

    public function delete(Request $request, User $user)
    {
        if (!$this->message->canDelete($user)) {
            return Response::FORBIDDEN;
        }

        return Messages::scrap($this->message);
    }

    public function edit(Request $request, User $user)
    {
        if (!$this->message->canEdit($user)) {
            return Response::FORBIDDEN;
        }

        $body = $request->put('body');

        if (empty($body)) {
            return Response::BAD_REQUEST;
        }

        return Messages::update(
            $this->message,
            [
                'body' => (string) $body,
            ]
        );
    }
}
