<?php

/*
 * This file is part of the ActiveCollab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

use ActiveCollab\Module\System\Events\DataObjectLifeCycleEvents\ConversationUserEvents\ConversationUserCreatedEvent;
use ActiveCollab\Module\System\Events\DataObjectLifeCycleEvents\ConversationUserEvents\ConversationUserUpdatedEvent;
use ActiveCollab\Module\System\Events\DataObjectLifeCycleEvents\MessageEvents\MessageCreatedEvent;

abstract class Conversation extends BaseConversation
{
    public function getMessages(User $user): CursorModelCollection
    {
        return Messages::prepareCursorCollection(
            sprintf(
                'conversation_messages_%s',
                $this->getId()
            ),
            $user
        );
    }

    public function createMessage(User $user, string $body): Message
    {
        $message = Messages::create(
            [
                'conversation_id' => $this->getId(),
                'body' => $body,
            ],
            false
        );

        $message->setCreatedBy($user);
        $message->save();

        DataObjectPool::announce(new MessageCreatedEvent($message));

        return $message;
    }

    public function newMessagesSince(User $user): ConversationUser
    {
        /** @var ConversationUser $conversation_user */
        $conversation_user = ConversationUsers::findOneBy(
            [
                'user_id' => $user->getId(),
                'conversation_id' => $this->getId(),
            ]
        );

        if (!$conversation_user) {
            $conversation_user = ConversationUsers::create(
                [
                    'user_id' => $user->getId(),
                    'conversation_id' => $this->getId(),
                    'new_messages_since' => new DateTimeValue(),
                ]
            );

            DataObjectPool::announce(new ConversationUserCreatedEvent($conversation_user));
        } else {
            $conversation_user->setNewMessagesSince(new DateTimeValue());
            $conversation_user->save();

            DataObjectPool::announce(new ConversationUserUpdatedEvent($conversation_user));
        }

        return $conversation_user;
    }

    public function whoCanSeeThis(): array
    {
        return $this->getMemberIds();
    }

    public function getMemberIdsPreloadKey(): string
    {
        return Conversation::class;
    }

    public function jsonSerialize(): array
    {
        return array_merge(
            parent::jsonSerialize(),
            [
                'parent_type' => $this->getParentType(),
                'parent_id' => $this->getParentId(),
                'messages' => Messages::getByConversation($this),
            ]
        );
    }

    public function validate(ValidationErrors &$errors)
    {
        if (!$this->validatePresenceOf('type')) {
            $errors->addError('Type filed is required.', 'type');
        }

        parent::validate($errors);
    }

    public function getRoutingContext(): string
    {
        return 'conversations';
    }

    public function getRoutingContextParams(): array
    {
        return [
            'id' => $this->getId(),
        ];
    }

    public function getUrlPath(): string
    {
        return '/conversations/' . $this->getId();
    }
}
