<?php

/*
 * This file is part of the ActiveCollab project.
 *
 * (c) A51 doo <info@activecollab.com>. All rights reserved.
 */

declare(strict_types=1);

use ActiveCollab\Module\System\Events\DataObjectLifeCycleEvents\MessageEvents\MessageUpdatedEvent;

class Messages extends BaseMessages
{
    public static function prepareCursorCollection(string $collection_name, User $user): CursorModelCollection
    {
        if ($user->isClient()) {
            throw new InvalidParamError('user', $user, '$user cannot be client.');
        }

        if (str_starts_with($collection_name, 'conversation_messages')) {
            return self::prepareConversationMessagesCollection($collection_name, $user);
        } else {
            throw new RuntimeException("Collection name '$collection_name' does not exist.");
        }
    }

    private static function prepareConversationMessagesCollection(string $collection_name, ?User $user)
    {
        $bits = explode('_', $collection_name);
        $conversation_id = (int) array_pop($bits);

        $conversation = Conversations::findById($conversation_id);

        if (empty($conversation)) {
            throw new ImpossibleCollectionError("Conversation #{$conversation_id} not found.");
        }

        if ($user && !$conversation->isMember($user)) {
            throw new ImpossibleCollectionError("User #{$user->getId()} is not a member of this conversation.");
        }

        $collection = parent::prepareCursorCollection($collection_name, $user);

        if (isset($_GET['cursor'])) {
            $collection->setCursor((int) $_GET['cursor']);
        }

        $collection->setConditions('conversation_id = ?', $conversation->getId());

        if (isset($_GET['limit'])) {
            $collection->setLimit((int) $_GET['limit']);
        }

        return $collection;
    }

    private static array $details_by_conversation = [];

    public static function getByConversation(Conversation $conversation): array
    {
        if (isset(self::$details_by_conversation[$conversation->getId()])) {
            return self::$details_by_conversation[$conversation->getId()];
        } else {
            $messages = self::find(
                [
                    'conditions' => [
                        'conversation_id = ?',
                        $conversation->getId(),
                    ],
                    'order' => 'id DESC',
                    'limit' => 20,
                ]
            );

            return $messages ? $messages->toArray() : [];
        }
    }

    public static function preloadDetailsByConversationIds(array $conversation_ids): void
    {
        $first_id = array_shift($conversation_ids);

        $query = "(SELECT id
                         FROM messages
                         WHERE conversation_id = {$first_id}
                         ORDER BY id DESC
                         LIMIT 20)";

        foreach ($conversation_ids as $conversation_id) {
            $query .= " UNION ALL (SELECT id
                         FROM messages
                         WHERE conversation_id = {$conversation_id}
                         ORDER BY id DESC
                         LIMIT 20)";
        }

        $ids = DB::executeFirstColumn($query);

        if ($ids) {
            /** @var Message[] $messages */
            $messages = Messages::findByIds($ids);

            foreach ($messages as $message) {
                if (!isset(self::$details_by_conversation[$message->getConversationId()])) {
                    self::$details_by_conversation[$message->getConversationId()] = [];
                }

                self::$details_by_conversation[$message->getConversationId()][] = $message->jsonSerialize();
            }
        }

        if ($zeros = array_diff([$first_id, ...$conversation_ids], array_keys(self::$details_by_conversation))) {
            foreach ($zeros as $conversation_with_no_message) {
                self::$details_by_conversation[$conversation_with_no_message] = [];
            }
        }
    }

    public static function &update(DataObject &$instance, array $attributes, bool $save = true): Message
    {
        $message = parent::update(
            $instance,
            array_merge(
                $attributes,
                [
                    'changed_on' => new DateTimeValue(),
                ]
            ),
            $save
        );

        DataObjectPool::announce(new MessageUpdatedEvent($message));

        $message->getConversation()->touch();

        return $message;
    }
}
